<?php
declare(strict_types=1);
require __DIR__ . '/bootstrap.php';

$data = json_input();
$assessmentId = isset($data['assessmentId']) ? (int)$data['assessmentId'] : 0;
if ($assessmentId <= 0) json_error('assessmentId is required');

$userId = current_user_id();

try {
    $pdo = db();
    // Close assessment
    $pdo->prepare('UPDATE assessments SET ended_at = NOW() WHERE id = ? AND user_id = ?')->execute([$assessmentId, $userId]);

    // Fetch totals
    $ass = $pdo->prepare('SELECT started_at, ended_at, total_easy, total_medium, total_hard FROM assessments WHERE id = ? AND user_id = ?');
    $ass->execute([$assessmentId, $userId]);
    $row = $ass->fetch();
    if (!$row) json_error('Assessment not found', 404);

    $totalEasy = (int)$row['total_easy'];
    $totalMedium = (int)$row['total_medium'];
    $totalHard = (int)$row['total_hard'];
    $totalQuestions = $totalEasy + $totalMedium + $totalHard;

    // Compute first-try correctness per question within the assessment
    // First attempt per (question_id) ordered by created_at
    $sql = 'SELECT a.question_id, a.difficulty, a.is_correct
            FROM attempts a
            WHERE a.assessment_id = ? AND a.user_id = ?
            AND a.id IN (
                SELECT MIN(id) FROM attempts WHERE assessment_id = a.assessment_id AND user_id = a.user_id AND question_id = a.question_id
            )';
    $st = $pdo->prepare($sql);
    $st->execute([$assessmentId, $userId]);
    $firsts = $st->fetchAll();

    $byDiff = [
        'easy' => ['correct' => 0, 'total' => $totalEasy],
        'medium' => ['correct' => 0, 'total' => $totalMedium],
        'hard' => ['correct' => 0, 'total' => $totalHard],
    ];
    $totalCorrect = 0;
    foreach ($firsts as $f) {
        $diff = $f['difficulty'];
        $corr = (int)$f['is_correct'] === 1;
        if (isset($byDiff[$diff]) && $corr) {
            $byDiff[$diff]['correct']++;
            $totalCorrect++;
        }
    }

    // Duration in seconds
    $started = strtotime((string)$row['started_at']);
    $ended = strtotime((string)$row['ended_at']);
    $duration = max(0, $ended - $started);

    json_ok([
        'totalCorrect' => $totalCorrect,
        'totalQuestions' => $totalQuestions,
        'byDifficulty' => $byDiff,
        'durationSeconds' => $duration,
    ]);
} catch (Throwable $e) {
    json_error('Failed to finish assessment: ' . $e->getMessage(), 500);
}

