<?php
// Basic bootstrap: session, DB (PDO), JSON helpers, and current user resolution

declare(strict_types=1);

// Start PHP session for anonymous user identification
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(204);
    exit;
}

function db(): PDO {
    static $pdo = null;
    if ($pdo) return $pdo;

    // Adjust to your Laragon/MySQL credentials
    $host = '127.0.0.1';
    $db   = 'adaptive_php';
    $user = 'root';
    $pass = '';
    $charset = 'utf8mb4';
    $dsn = "mysql:host=$host;dbname=$db;charset=$charset";
    $options = [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        PDO::ATTR_EMULATE_PREPARES => false,
    ];
    try {
        $pdo = new PDO($dsn, $user, $pass, $options);
    } catch (Throwable $e) {
        json_error('DB connection failed: ' . $e->getMessage(), 500);
    }
    return $pdo;
}

function json_input(): array {
    $raw = file_get_contents('php://input');
    if ($raw === false || $raw === '') return [];
    $data = json_decode($raw, true);
    return is_array($data) ? $data : [];
}

function json_ok($payload, int $status = 200): void {
    http_response_code($status);
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode(['ok' => true, 'data' => $payload], JSON_UNESCAPED_UNICODE);
    exit;
}

function json_error(string $message, int $status = 400): void {
    http_response_code($status);
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode(['ok' => false, 'error' => $message], JSON_UNESCAPED_UNICODE);
    exit;
}

function require_admin(): int {
    $aid = $_SESSION['account_id'] ?? null;
    if (!$aid) {
        json_error('Unauthorized', 401);
    }
    $pdo = db();
    $st = $pdo->prepare('SELECT id FROM accounts WHERE id = ? AND role = "admin"');
    $st->execute([(int)$aid]);
    $row = $st->fetch();
    if (!$row) {
        json_error('Forbidden', 403);
    }
    return (int)$aid;
}

function current_user_id(): int {
    $pdo = db();

    $sid = session_id();
    $accountId = $_SESSION['account_id'] ?? null;
    if ($accountId) {
        // Nếu session_id đã tồn tại ở user khác, xóa session_id đó trước
        $stmt = $pdo->prepare('SELECT id FROM users WHERE session_id = ? AND (account_id IS NULL OR account_id != ?) LIMIT 1');
        $stmt->execute([$sid, (int)$accountId]);
        $dupRow = $stmt->fetch();
        if ($dupRow) {
            $upd = $pdo->prepare('UPDATE users SET session_id = NULL WHERE id = ?');
            $upd->execute([(int)$dupRow['id']]);
        }
        // Tìm theo account_id trước
        $stmt = $pdo->prepare('SELECT id FROM users WHERE account_id = ? LIMIT 1');
        $stmt->execute([(int)$accountId]);
        $row = $stmt->fetch();
        if ($row) {
            // Nếu đã có user, cập nhật session_id
            $upd = $pdo->prepare('UPDATE users SET session_id = ? WHERE id = ?');
            $upd->execute([$sid, (int)$row['id']]);
            return (int)$row['id'];
        }
        // Nếu chưa có, tạo mới
        $ins = $pdo->prepare('INSERT INTO users (session_id, account_id, created_at) VALUES (?, ?, NOW())');
        $ins->execute([$sid, (int)$accountId]);
        return (int)$pdo->lastInsertId();
    }
    // Anonymous session fallback
    $stmt = $pdo->prepare('SELECT id FROM users WHERE session_id = ?');
    $stmt->execute([$sid]);
    $row = $stmt->fetch();
    if ($row && isset($row['id'])) return (int)$row['id'];
    $ins = $pdo->prepare('INSERT INTO users (session_id, created_at) VALUES (?, NOW())');
    $ins->execute([$sid]);
    return (int)$pdo->lastInsertId();
}
