<?php
declare(strict_types=1);
require __DIR__ . '/bootstrap.php';

$in = json_input();
$code = (string)($in['code'] ?? '');
if ($code === '') json_error('code is required');

// Strip surrounding tags if present
$code = preg_replace('/^\s*<\?php/i', '', $code);
$code = preg_replace('/\?>\s*$/', '', $code);

// Build a temporary file
$tmpDir = __DIR__ . '/tmp';
if (!is_dir($tmpDir)) @mkdir($tmpDir, 0775, true);
$file = tempnam($tmpDir, 'run_');
if ($file === false) json_error('Cannot create temp file', 500);

$payload = "<?php\n" . $code . "\n";
file_put_contents($file, $payload);

// Prepare command with defensive flags (CLI)
$php = 'php';
$flags = [
    '-d', 'display_errors=1',
    '-d', 'log_errors=0',
    // Try to disable dangerous functions (may require php.ini, best effort)
    '-d', 'disable_functions=exec,passthru,shell_exec,system,proc_open,popen,curl_exec,curl_multi_exec,parse_ini_file,show_source,pcntl_exec',
    // Reduce memory for safety
    '-d', 'memory_limit=64M',
];
$cmd = $php . ' ' . implode(' ', array_map('escapeshellarg', $flags)) . ' ' . escapeshellarg($file);

$descriptor = [
    1 => ['pipe', 'w'], // stdout
    2 => ['pipe', 'w'], // stderr
];
$proc = proc_open($cmd, $descriptor, $pipes, __DIR__);
if (!\is_resource($proc)) {
    @unlink($file);
    json_error('Failed to start PHP process', 500);
}

// Enforce short timeout
$timeout = 2.0; // seconds
$start = microtime(true);
$stdout = '';
$stderr = '';

stream_set_blocking($pipes[1], false);
stream_set_blocking($pipes[2], false);

while (true) {
    $stdout .= stream_get_contents($pipes[1]) ?: '';
    $stderr .= stream_get_contents($pipes[2]) ?: '';
    $status = proc_get_status($proc);
    $elapsed = microtime(true) - $start;
    if (!$status['running']) break;
    if ($elapsed > $timeout) {
        // Kill
        proc_terminate($proc, 9);
        $stderr .= "\n[terminated: timeout]";
        break;
    }
    usleep(10000);
}

fclose($pipes[1]);
fclose($pipes[2]);
proc_close($proc);
@unlink($file);

// Truncate long outputs
if (strlen($stdout) > 5000) $stdout = substr($stdout, 0, 5000) . "\n... (truncated)";
if (strlen($stderr) > 5000) $stderr = substr($stderr, 0, 5000) . "\n... (truncated)";

json_ok(['stdout' => $stdout, 'stderr' => $stderr]);

